<?php
/**
 * Logger Trait
 * Adds custom logging features to plugins
 *
 * @author Gabriel Jenik <http://www.encuesta.biz/>
 * @copyright 2022 Gabriel Jenik <http://www.encuesta.biz/>
 * @license Propietary
 * @version 2.1.0
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
namespace CalculateCompleteness\Traits;

trait Logger {

    protected $logDirectory;
    protected $htmlEnabled = false;
    protected $outputEnabled = false;

    protected function initLogger($logDirectory = null)
    {
        $this->logDirectory = $logDirectory;
        if (empty($this->logDirectory)) {
            $this->logDirectory = \Yii::app()->getConfig('tempdir') . '/runtime/logs';
        }
        // Create log folder if it doesn't exist
        if (!file_exists($this->logDirectory)) {
            mkdir($this->logDirectory, 0777, true); 
        }
    }

    /**
     *  Output formatted message
     */
    protected function output($message)
    {
        if ($this->outputEnabled) {
            if ($this->htmlEnabled) {
                echo str_replace("\n","<br/>",$message)."<br/>";
                flush();
            } else {
                echo $message."\n";
            }
        }
    }

    /**
     *  Logs a message in the local log file
     */
    protected function logMsg($message, $level = 'INFO')
    {
        if (!$this->aboveLogLevel($level)) {
            return;
        }

        $timestamp = dateShift(date("Y-m-d H:i:s"), "Y-m-d H:i:s", \Yii::app()->getConfig("timeadjust"));
        $levelPart = "[".$level."]\t";
        $surveyPart = !empty($this->surveyId) ? "[SID:{$this->surveyId}]\t" : "";
        $message = $timestamp.":\t" . $levelPart . $surveyPart . str_pad($message, 10);

        // Output message
        $this->output($message);

        if (empty($this->logDirectory)) {
            return;
        }

        // Log to file
        $logFileName = $this->logDirectory . DIRECTORY_SEPARATOR . $this->getName() . "_" . date("Y-m-d") . ".log";
        $log = fopen($logFileName, 'a');
        fwrite($log, $message."\n");  
        fclose($log);
    }

    /**
     *  Logs a message in the local log file if debug is enabled
     */
    protected function debugMsg($message)
    {
        $this->logMsg($message, 'DEBUG');
    }

    protected function getLogLevel()
    {
        $level = 'DEBUG';
        if (!empty($this->logLevel)) {
            $level = $this->logLevel;
        }
        return strtoupper($level);
    }

    protected function setLogLevel($level)
    {
        $this->logLevel = strtoupper($level);
    }

    protected function aboveLogLevel($outputLevel)
    {
        $levels = [
            'TRACE' => 0,
            'DEBUG' => 1,
            'INFO' => 2,
            'WARN' => 3,
            'WARNING' => 3,
            'ERROR' => 4,
            'FATAL' => 5
        ];
        $loggerLevel = $this->getLogLevel();
        $loggerLevelNum = isset($levels[$loggerLevel]) ? $levels[$loggerLevel] : -999;
        $outputLevel = strtoupper($outputLevel);
        $outputLevelNum = isset($levels[$outputLevel]) ? $levels[$outputLevel] : 999;
        return $outputLevelNum >= $loggerLevelNum;
    }

    /**
     * Override default log method
     * @inheritdoc
     */
    public function log($message, $level = 'INFO')
    {
        $this->logMsg($message, $level);
    }
}