<?php

/**
 * IPRangeSurvey Plugin.
 *
 * Installation:
 * 1 - Copy the IPRangeSurvey folder into the "plugins" directory of Limesurvey.
 * 2 - Go to Configuration > Plugin Manager.
 * 3 - Find the entry related to the plugin and activate it.
 */

class IPRangeSurvey extends PluginBase
{
    static protected $description = 'IPRangeSurvey';
    static protected $name = 'IPRangeSurvey';
    static protected $copy_response = '';

    protected $storage = 'DbStorage';

    public function init()
    {
        $this->subscribe('afterPluginLoad');
        $this->subscribe('beforeSurveyPage');
        $this->subscribe('beforeSurveySettings');
        $this->subscribe('newSurveySettings');
    }

    public function beforeSurveyPage()
    {
        $event      = $this->getEvent();
        $surveyId   = $event->get('surveyId');

        $ipRange    = $this->get('range', 'Survey', $surveyId);
        $redirectUrl = $this->get('url','Survey', $surveyId);

        if (!empty($ipRange)) {
            $ip = $_SERVER['REMOTE_ADDR'];
            if (!$this->inRange($ip, $ipRange)) {
                Yii::app()->request->redirect($redirectUrl);
            }
        }
    }

    public function beforeSurveySettings()
    {
        $this->getEvent()->set("surveysettings.{$this->id}", array(
            'name' => get_class ($this),
            'settings' => array(
                'range' => array(
                    'type' => 'string',
                    'label' => $this->t('Allowed IP Range'),
                    'current' => $this->get('range', 'Survey', $this->event->get('survey'),''),
                ),
                'url' => array(
                    'type' => 'string',
                    'label' => $this->t('Redirect URL'),
                    'current' => $this->get('url', 'Survey', $this->event->get('survey'),''),
                )
            ))
        );
    }

    public function newSurveySettings()
    {
        $event = $this->event;
        foreach ($event->get('settings') as $name => $value) {
            $this->set($name, $value, 'Survey', $event->get('survey'), '');
        }
    }

    function inRange($ip, $range)
    {
        if (strpos($range, '/') == false) {
            $range .= '/32';
        }

        // $range is in IP/CIDR format eg 127.0.0.1/24
        list($range, $netmask) = explode( '/', $range, 2);

        $rangeDecimal = ip2long($range);
        $ipDecimal    = ip2long($ip);

        $wildcardDecimal = pow(2, (32 - $netmask )) - 1;
        $netmaskDecimal = ~ $wildcardDecimal;

        return (($ipDecimal & $netmaskDecimal) == ($rangeDecimal & $netmaskDecimal));
    }

    public function afterPluginLoad()
    {
        Yii::app()->setComponent('ipRangeSurveyMessages', array(
            'class' => 'CPhpMessageSource',
            'basePath' => __DIR__ . DIRECTORY_SEPARATOR . 'messages',
            'forceTranslation' => false,
            'onMissingTranslation' => function ($event) {
                var_dump($event->message); //exit;
            }
        ));
    }

    public function t($message)
    {
        return Yii::t('lang', $message, array(), 'ipRangeSurveyMessages');
    }

}
