<?php
/**
 * AssetHandler Trait
 * Adds asset handling features to plugins
 *
 * @author Gabriel Jenik <http://www.encuesta.biz/>
 * @copyright 2022 Gabriel Jenik <http://www.encuesta.biz/>
 * @license Propietary
 * @version 1.0.0
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
namespace CalculateCompleteness\Traits;

trait AssetHandler {

    /**
     * Treat a url path and decide if have to append the app path or not.
     */
    protected static function assetTreatPath($src, $pathPrefix)
    {
        // If this is an array with a file bucket, process that file.
        if (is_array($src) && !empty($src['file']))
        {
            $src['file'] = static::assetTreatPath($src['file'], $pathPrefix);
            return $src;
        }

        // If not scalar, maybe an array, just bypass
        if (!is_scalar($src)) return $src;

        // Don't apply absolute path if $src starts https://
        if (substr($src, 0, 8) == "https://") return $src;

        // Don't apply absolute path if $src starts //
        if (substr($src, 0, 2) == "//") return $src;

        // Don't apply absolute path if $src starts /
        if (substr($src, 0, 1) == "/") return $src;

        // Don't apply absolute path if $src is relative
        if (substr($src, 0, 1) == ".") return $src;

        // Apply pathRoot
        $outSrc = $pathPrefix . "/" . $src;

        // Remove initial "//" if resulted in that.
        if (substr($outSrc, 0, 2) == "//") $outSrc = substr($outSrc, 1);

        return $outSrc;
    }

    /**
     * Publish assets from a plugin
     */
    protected function publishAsset($asset, $plugin = NULL)
    {
        return \Yii::app()->assetManager->publish($this->getAssetDir($asset, $plugin) );
    }

    /**
     * Register an asset list
     */
    protected function registerAssets($assetList, $assetUrl, $extraParams = [])
    {
        foreach ($assetList as $method => $assets)
        {
            foreach ($assets as $file)
            {
                $params = static::assetTreatPath($file, $assetUrl);
                if (!is_array($params)) $params = [$params];

                // Extra Params
                if ($method == "registerScriptFile" && !empty($extraParams["position"]))
                {
                    $params[1] = $extraParams["position"];
                }

                call_user_func_array(
                    [App()->getClientScript(), $method],
                    $params
                );
            }
        }
    }

    /**
     * Returns a plugin base url
     */
    protected function getPluginUrl($class = NULL)
    {
        $path = $this->getPluginDir($class);
        if (stripos($path, ROOT) === 0) {
            $path = substr($path, strlen(ROOT));
        }
        $subfolder = str_replace("\\", "/", $path);
        $url = \Yii::app()->request->getBaseUrl() . $subfolder;
        return $url;
    }

    protected function getAssetUrl($asset, $class = NULL)
    {
        return $this->getPluginUrl($class) . "/assets/" . $asset . "/";
    }

    protected function getPluginDir($class = NULL)
    {
        if (empty($class)) {
            $object = new \ReflectionObject($this);
        } else {
            $object = new \ReflectionClass($class);
        }

        $filename = $object->getFileName();
        $basePath = dirname($filename);

        return $basePath;
    }

    protected function getPluginBaseUrl()
    {
        $pluginDir = $this->getPluginDir();
        $pluginDir = str_replace(FCPATH, "", $pluginDir);
        $url = \Yii::app()->getConfig('publicurl') . $pluginDir;
        return $url;
    }

    protected function getPluginFileUrl($relativePath)
    {
        return $this->getPluginBaseUrl() . '/' . $relativePath;
    }

    protected function getAssetDir($asset, $class = NULL)
    {
        return $this->getPluginDir($class) . "/assets/" . $asset . "/";
    }

    /**
     * Get Current LS Version
     */
    public function LSVersion()
    {
        return \Yii::app()->getConfig('versionnumber');
    }

    /**
     * Compare Current LS Version with requested
     */
    public function LSVersionCompare($version, $compare = ">=")
    {
        return version_compare($this->LSVersion(), $version, $compare);
    }
}