<?php
/**
 * Multilanguage Trait
 * Adds message translation features to plugins
 *
 * @author Gabriel Jenik <http://www.encuesta.biz/>
 * @copyright 2022 Gabriel Jenik <http://www.encuesta.biz/>
 * @license Propietary
 * @version 2.0.0
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
namespace CalculateCompleteness\Traits;

trait Multilanguage {

    protected $defaultLanguage;
    protected $translationMode = 'by-key';

    /**
     *  Gets the requested translation from the language resource file.
     *  
     *  If the specified language is not found in the resource file,
     *  the method tries to default to 'en'.
     */
    public function getTranslation($lang, $msg, $byKey = false)
    {
        if (!isset($this->defaultLanguage)) $this->defaultLanguage = 'en';
        if (!isset($this->_locs)) {
            if (isset($this->langResourceFile)) {
                $this->_locs = require $this->langResourceFile;
            } else {
                $this->_locs = self::loadAllTranslations();
            }
        }

        if (!$byKey && $this->translationMode == 'by-message') {
            $key = $this->getTranslationKey($msg);
        } else {
            $key = $msg;
        }

        if (!array_key_exists($lang, $this->_locs)) {
            if ($lang!=$this->defaultLanguage) {
                return $this->getTranslation($this->defaultLanguage, $key, true);
            } else {
                // Throw exception if debug
                if (!empty($this->debug)) throw new \Exception("Language \"{$lang}\" was not found on resource file.", 1);
                // Return empty if not.
                return $msg;
            }
        }

        if (!array_key_exists($key, $this->_locs[$lang]))
        {
            if ($lang!=$this->defaultLanguage) {
                return $this->getTranslation($this->defaultLanguage, $key, true);
            } else {
                // Throw exception if debug
                if (!empty($this->debug)) throw new \Exception("\"{$msg}\" key was not found on resource file for language \"{$lang}\"", 1);
                // Return empty if not.
                return $msg;
            }
        }

        return $this->_locs[$lang][$key];
    }

    protected function getTranslationKey($msg)
    {
        if (!isset($this->_locs[$this->defaultLanguage])||!is_array($this->_locs[$this->defaultLanguage])) {
            // Throw exception if debug
            if (!empty($this->debug)) throw new \Exception("Default language \"{$this->defaultLanguage}\" was not found on resource file.", 1);
            // Return empty if not.
            return $msg;
        }
        foreach ($this->_locs[$this->defaultLanguage] as $key => $resMsg) {
            if (strtolower($msg) == strtolower($resMsg)) return $key;
        }
        return $msg;
    }

    /**
     *  Get translated message.
     *  
     *  Params: language, message key, aditional arguments
     */
    public function translateTo() {
        // Parse arguments
        $args = func_get_args();
        if (count($args) < 2) {
            return;
        }
        $lang = array_shift($args);
        $msg = array_shift($args);

        // Get translations
        $translation = $this->getTranslation($lang, $msg);

        // If no more arguments left (just one original argument), no translation
        if (count($args) == 0)
        {
            $sprintf_args = [];
            $finalMessage = $translation;
        }
        else
        {
            // Prepare final message
            $sprintf_args = $args;
            array_unshift($sprintf_args, $translation);
            $finalMessage = call_user_func_array("sprintf", $sprintf_args);
        }
        return $finalMessage;
    }

    /**
     *  Get translated message usin app language
     *  
     *  Params: message key, aditional arguments
     */
    public function translate() {
        $args = func_get_args();
        $lang = !empty($this->language) ? $this->language : (isset(\Yii::app()->session['adminlang']) ? \Yii::app()->session['adminlang'] : \Yii::app()->getLanguage());
        array_unshift($args, $lang);
        return call_user_func_array([$this, 'translateTo'], $args);
    }

    /**
     * Set the translation mode by message instead of by key
     */
    protected function setTranslationModeByMessage()
    {
        $this->translationMode = 'by-message';
    }

    protected function setLangFromResponse($surveyId, $responseId)
    {
        try {
            $response = \Response::model($surveyId)->findByPk($responseId);
        } catch (\Exception $ex) {
            // Nothing to do
        }
        $this->language = !empty($response) ? $response->startlanguage : '';
    }

    protected static function loadAllTranslations()
    {
        $baseClass = get_class();
        $currentClass = get_called_class();
        if ($baseClass != $currentClass) {
            $parentClass = get_parent_class($currentClass);
            return self::mergeTranslations(call_user_func($parentClass . "::loadAllTranslations"), self::loadSelfTranslations());
        } else {
            return self::loadSelfTranslations();
        }
    }

    protected static function loadSelfTranslations()
    {
        $langResource = [];
        $resourceFile = self::_getDir() . '/assets/messages.php';
        if (file_exists($resourceFile)) {
            $langResource = require $resourceFile;
        }
        return $langResource;
    }

    protected static function mergeTranslations($baseResource, $extendedResource)
    {
        $mergedResource = [];
        foreach ($baseResource as $lang => $baseTranslations) {
            if (!empty($extendedResource[$lang]) && is_array($extendedResource[$lang])) {
                $mergedResource[$lang] = array_merge($baseTranslations, $extendedResource[$lang]);
            } else {
                $mergedResource[$lang] = $baseTranslations;
            }
            unset($extendedResource[$lang]);
        }
        $mergedResource = array_merge($mergedResource, $extendedResource);
        return $mergedResource;
    }

    protected static function _getDir()
    {
        $reflClass = new \ReflectionClass(get_called_class());
        $fileName = $reflClass->getFileName();
        return dirname($fileName);
    }
}