<?php

/**
 * CalculateCompleteness
 * Calculates Survey summary to show on last page
 *
 * @author Gabriel Jenik <gabriel@jenik.com.ar>
 * @copyright 2022 Gabriel Jenik Encuesta.Biz
 * @version 1.0.6
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

require_once __DIR__ . '/vendor/autoload.php';

use CalculateCompleteness\Traits\Multilanguage;
use CalculateCompleteness\Traits\Logger;
use CalculateCompleteness\Traits\AssetHandler;

class CalculateCompleteness extends PluginBase
{
    use Multilanguage;
    use Logger;
    use AssetHandler;

    protected $storage = 'DbStorage';
    static protected $description = 'CalculateCompleteness';
    static protected $name = 'CalculateCompleteness';

    static protected $instance = null;
    protected $summaryQid = null;
    protected $indexItems = [];
    protected $summaryQuestions = [];
    protected $questionAttributes = [];
    protected $sgqs = [];
    protected $fieldmap = [];

    const DEFAULT_LAYOUT = 'qp';
    const QUESTION_TYPES_MUST_BE_FULL = [
        'F', // Array
        'A', // Array (5 point choice)
        'B', // Array (10 point choice)
        'E', // Array (Increase/Same/Decrease)
        ':', // Array (Numbers)
        ';', // Array (Texts)
        'C', // Array (Yes/No/Uncertain)
        'H', // Array by column
        '1', // Array dual scale
        'Q', // Multiple short text
        'K', // Multiple numerical input
    ];

    public function init()
    {
        $this->initLogger();
        $this->subscribe("beforeQuestionRender");
        $this->subscribe("beforeSurveyPage");
        $this->subscribe('newQuestionAttributes');
        $this->subscribe('newDirectRequest');
        $this->subscribe('beforeSurveySettings');
        $this->subscribe('newSurveySettings');
    }

    /**
     * Survey Settings
     */
    public function beforeSurveySettings()
    {
        $event = $this->event;
        $surveyId = $event->get("survey");
        
        $newSettings = [
            /*'activate' => [
                'type' => 'boolean',
                'label' => $this->translate('setting.activate.label'),
                'help' => $this->translate('setting.activate.help'),
                'current' => $this->get('activate', 'Survey', $surveyId, false)
            ],*/
            'showLink' => [
                'type' => 'boolean',
                'label' => $this->translate('setting.showLink.label'),
                'help' => $this->translate('setting.showLink.help'),
                'current' => $this->get('showLink', 'Survey', $surveyId, false)
            ],
        ];

        // Set all settings
        $event->set("surveysettings.{$this->id}", array(
            'name' => get_class($this),
            'settings' => $newSettings,
        ));
    }

    /**
     * Handle Survey Settings Saving
     */
    public function newSurveySettings()
    {
        $event = $this->event;
        foreach ($event->get('settings') as $name => $value)
        {
            $this->set($name, $value, 'Survey', $event->get('survey'));
        }
    }

    public function beforeQuestionRender() {
        $event = $this->getEvent();
        $surveyId = $event->get('surveyId');
        $qid = $event->get('qid');

        $includeSummary = $this->getQuestionAttribute('includeSummary', $qid, false);
        if (!$includeSummary) {
            return;
        }

        if (!isset($_SESSION['survey_' . $surveyId]['srid'])) {
            return;
        }
        $responseId = $_SESSION['survey_' . $surveyId]['srid'];
        $summaryHtml = $this->renderSummary($surveyId, $responseId, $qid, true);

        $text = $event->get('text');
        $event->set('text', $text . $summaryHtml);
    }

    public function newQuestionAttributes()
    {
        $event = $this->getEvent();
        $questionAttributes = [
            'includeSummary' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 10,
                'inputtype' => 'switch',
                'default'   => 0,
                'help'      => $this->translate('question_attribute.show.help'),
                'caption'   => $this->translate('question_attribute.show.caption'),
                'expression'=> [],
                'options'   => [
                    'option' => [
                        [
                            'value' => 0,
                            'text' => gT('No')
                        ],
                        [
                            'value' => 1,
                            'text' => gT('Yes')
                        ]
                    ]
                ],
            ],
            'summaryLayout' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 20,
                'inputtype' => 'singleselect',
                'default'   => self::DEFAULT_LAYOUT,
                'help'      => $this->translate('question_attribute.layout.help'),
                'caption'   => $this->translate('question_attribute.layout.caption'),
                'expression'=> [],
                'options'   => [
                    'qp' => $this->translate('question_attribute.layout.option.qp'), // Incomplete List + Progress
                    'pq' => $this->translate('question_attribute.layout.option.pq'), // Progress + Incomplete List
                    'q' => $this->translate('question_attribute.layout.option.q'), // Incomplete List
                    'p' => $this->translate('question_attribute.layout.option.p') // Progress
                ],
            ],
            'summaryListDescription' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 30,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.list_description.help'),
                'caption'   => $this->translate('question_attribute.list_description.caption'),
                'i18n'      => true,
            ],
            'summaryProgressDescription' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 40,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.progress_description.help'),
                'caption'   => $this->translate('question_attribute.progress_description.caption'),
                'i18n'      => true,
            ],
            'summaryCompleteDescription' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 50,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.complete_message.help'),
                'caption'   => $this->translate('question_attribute.complete_message.caption'),
                'i18n'      => true,
            ],
            'summaryIncompleteWarning' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 60,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.incomplete_warning.help'),
                'caption'   => $this->translate('question_attribute.incomplete_warning.caption'),
                'i18n'      => true,
            ],
            'summaryExcludeQuestions' => [
                'types'     => 'X',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 70,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.exclude_questions.help'),
                'caption'   => $this->translate('question_attribute.exclude_questions.caption'),
            ],
            'alternativeText' => [
                'types'     => '15ABCDEFGHIKLMNOPQRSTUY!:;|',
                'category'  => $this->translate('question_attribute.category'),
                'sortorder' => 10,
                'inputtype' => 'textarea',
                'default'   => '',
                'help'      => $this->translate('question_attribute.alternative_text.help'),
                'caption'   => $this->translate('question_attribute.alternative_text.caption'),
                'i18n'      => true,
            ],
        ];
        $event->append('questionAttributes', $questionAttributes);
    }

    /**
     * Subscription to newDirectRequest Request event
     */
    public function newDirectRequest()
    {
        $event = $this->event;

        if ($event->get('target') != $this->getName()) return;

        // Get parameters
        $request = Yii::app()->request;
        $surveyId = $request->getParam('surveyId', '');

        $function = $request->getParam('function');

        switch ($function) {
            case "check":
                $responseId = $request->getParam('srid', '');
                $this->getSummaryAjax($surveyId, $responseId);
                break;
            default:
                $this->logMsg("No function specified.");
        }
    }

    private function getSummaryAjax($surveyId, $responseId)
    {
        $response = [];
        try {
            $html = $this->renderSummary($surveyId, $responseId);
            if (!empty($html)) {
                $response = [
                    'success' => true,
                    'html' => $html
                ];
            } else {
                $response = [
                    'success' => false,
                    'error' => 'Error rendering the summary'
                ];
            }
        } catch (Throwable $ex) {
            $response = [
                'success' => false,
                'error' => $ex->getMessage()
            ];
        }
        echo json_encode($response);
    }

    private function renderSummary($surveyId, $responseId, $qid = null, $includeIncompleteWarning = false)
    {
        if (empty($surveyId) || empty($responseId)) {
            return;
        }
        if (is_null($qid)) {
            $qid = $this->getSummaryQuestion($surveyId);
        }
        if (empty($qid)) {
            return;
        }
        $this->summaryQid = $qid;

        $excludes = $this->getExcludedQuestions($qid);
        $summaryData = $this->calculateSummaryData($surveyId, $responseId, $excludes);
        $summaryHtml = $this->getSummaryHtml($summaryData);

        if (
            $includeIncompleteWarning
            && !empty($summaryData['classifiedQuestions']['empty'])
        ) {
            $summaryHtml .= $this->getQuestionAttribute('summaryIncompleteWarning', null, $this->translate('question_attribute.incomplete_warning.default'));
            $summaryHtml .= $this->getIncompleteScript();
        }

        return $summaryHtml;
    }

    private function calculateSummaryData($surveyId, $responseId, $excludes = [])
    {
        $classified = $this->getClassifiedQuestions($surveyId, $responseId, $excludes);
        $skipped = !empty($classified['skipped']) ? count($classified['skipped']) : 0;
        $complete = !empty($classified['complete']) ? count($classified['complete']) : 0;
        $empty = !empty($classified['empty']) ? count($classified['empty']) : 0;
        $progress = ($empty + $complete != 0) ? $complete / ($empty + $complete) : 0;

        $indexMode = '';
        $survey = Survey::model()->findByPk($surveyId);
        if ($survey->aOptions['questionindex'] == 1 || $survey->aOptions['questionindex'] == 2) {
            if ($survey->aOptions['format'] == 'G') {
                $indexMode = 'G';
            } elseif ($survey->aOptions['format'] == 'S') {
                $indexMode = 'Q';
            }
        }

        return [
            'classifiedQuestions' => $classified,
            'skipped' => $skipped,
            'complete' => $complete,
            'empty' => $empty,
            'progress' => $progress,
            'indexMode' => $indexMode,
        ];
    }

    private function getClassifiedQuestions($surveyId, $responseId, $excludes = [])
    {
        $response = $this->pluginManager->getApi()->getResponse($surveyId, $responseId, false);
        if (empty($response)) {
            return [];
        }

        $language = Yii::app()->language;
        if (!empty(Yii::app()->session['survey_' . $surveyId]['s_lang'])) {
            $language = Yii::app()->session['survey_' . $surveyId]['s_lang'];
        }
        $survey = Survey::model()->findByPk($surveyId);
        if (!in_array($language, $survey->allLanguages)) {
            $language = $survey->language;
        }

        $sgqsByQuestion = $this->getSGQsByQuestion($surveyId);
        $relevanceStatus = Yii::app()->session['survey_' . $surveyId]['relevanceStatus'];

        $questions = [];
        foreach ($sgqsByQuestion as $title => $sgqs) {
            if (in_array(strtolower($title), $excludes)) {
                continue;
            }
            $skipped = false;
            $question = Question::model()->findByAttributes(['title' => $title, 'sid' => $surveyId]);
            $alternativeText = $this->getQuestionAttribute('alternativeText', $question->qid, '', $language);
            $questionText = !empty($alternativeText) ? $alternativeText : $question->questionl10ns[$language]->question;
            $questionDetails = [
                'title' => $title,
                'sgqs' => $sgqs,
                'question' => $questionText,
                'group' => $question->group->questiongroupl10ns[$language]->group_name,
                'qid' => $question->qid,
                'gid' => $question->group->gid,
            ];
            if (
                (isset($relevanceStatus[$question->gid]) && !$relevanceStatus[$question->gid])
                || (isset($relevanceStatus[$question->qid]) && !$relevanceStatus[$question->qid])
            ) {
                $skipped = true;
            } else {
                $skipped = true;
                foreach ($sgqs as $sgq) {
                    if (!isset($relevanceStatus[$sgq]) || $relevanceStatus[$sgq]) {
                        $skipped = false;
                        break;
                    }
                }
            }
            if ($skipped) {
                $questions['skipped'][] = $questionDetails;
                continue;
            }

            if ($this->isGenericAnswerQuestion($question)) {
                $answer = strtolower($response[$sgqs[0]]);
                /* ORIGINAL LOGIC
                $comment = $response[$sgqs[1]];
                if (
                    $answer == 'yes'
                    || $answer == 'no'
                    || ($answer == 'not relevant' && !empty($comment))
                ) {
                    $questions['complete'][] = $questionDetails;
                } else {
                    $questions['empty'][] = $questionDetails;
                }*/

                // Simplified logic
                if (isset($answer) && strlen($answer) > 0) {
                    $questions['complete'][] = $questionDetails;
                } else {
                    $questions['empty'][] = $questionDetails;
                }

                continue;
            }

            $allEmpty = true;
            $anyEmpty = false;
            foreach ($sgqs as $sgq) {
                // Skip irrelevant subquestions
                if (isset($relevanceStatus[$sgq]) && !$relevanceStatus[$sgq]) {
                    continue;
                }
                $value = $response[$sgq];
                if (is_null($value) || (is_string($value) && strlen($value) == 0)) {
                    $anyEmpty = true;
                } else {
                    $allEmpty = false;
                }
            }
            $empty = $this->allSubquestionsMustBeCompleted($question) ? $anyEmpty : $allEmpty;

            if (!$empty) {
                $questions['complete'][] = $questionDetails;
            } else {
                $questions['empty'][] = $questionDetails;
            }
        }
        return $questions;
    }

    protected function allSubquestionsMustBeCompleted($question)
    {
        return in_array($question->type, self::QUESTION_TYPES_MUST_BE_FULL);
    }

    protected function isGenericAnswerQuestion($question)
    {
        if ($question->type != ';') {
            return false;
        }
        $cssClasses = $this->getQuestionAttribute('cssclass', $question->qid, '');
        return in_array('GenericAnswer', explode(" ", $cssClasses));
    }

    protected function getSGQsByQuestion($surveyId)
    {
        $questions = [];
        $sgqs = $this->getSGQs($surveyId);
        $fieldmap = $this->getFielmap($surveyId);
        foreach ($sgqs as $title => $sgq) {
            if (!empty($fieldmap[$sgq]['type']) && $fieldmap[$sgq]['type'] == 'X') {
                continue;
            }
            $question = explode("_", $title)[0];
            $questions[$question][] = $sgq;
        }
        return $questions;
    }

    protected function getSGQs($surveyId, $toLowerCase = false)
    {
        if (empty($this->sgqs[$surveyId])) {
            $fieldmap = $this->getFielmap($surveyId);

            $sgqs = [];
            foreach ($fieldmap as $fieldname => $fielddata) {
                if (!empty($fielddata['title']) && !empty($fielddata['qid'])) {
                    $key = empty($fielddata['aid']) ? $fielddata['title'] : $fielddata['title'] . '_' . $fielddata['aid'];
                    if ($toLowerCase) {
                        $key = strtolower($key);
                    }
                    $sgqs[$key] = $fieldname;
                }
            }
            $this->sgqs[$surveyId] = $sgqs;
        }
        return $this->sgqs[$surveyId];
    }

    protected function getFielmap($surveyId)
    {
        if (empty($this->fieldmap[$surveyId])) {
            $oSurvey = Survey::model()->findByPk($surveyId);
            $this->fieldmap[$surveyId] = createFieldMap($oSurvey, 'full', false);
        }
        return $this->fieldmap[$surveyId];
    }

    private function getSummaryHtml($summaryData)
    {
        $html = '';

        $layout = $this->getQuestionAttribute('summaryLayout', null, self::DEFAULT_LAYOUT);
        $parts = preg_split("//", $layout);
        foreach ($parts as $part) {
            switch ($part) {
                case 'q':
                    $html .= $this->getSummaryTable($summaryData);
                    break;
                case 'p':
                    $html .= $this->getProgressHtml($summaryData);
                    break;
            }
        }

        return $html;
    }

    private function getSummaryTable($summaryData)
    {
        $html = '';

        if (!empty($summaryData['classifiedQuestions']['empty'])) {
            $description = $this->getQuestionAttribute('summaryListDescription', null, $this->translate('question_attribute.list_description.default'));
            $html .=
                '<div class="summary-empty-questions">' .
                    '<div class="summary-empty-questions-description">' .$description . '</div>' .
                    '<table class="table table-bordered table-hover">' .
                        '<thead>' .
                            '<tr class="ls-heading">' .
                                '<th>' . $this->translate('question_list.header.group') . '</th>' .
                                '<th>' . $this->translate('question_list.header.question') . '</th>' .
                            '</tr>' .
                        '</thead>' .
                        '<tbody>';
            foreach ($summaryData['classifiedQuestions']['empty'] as $questionDetails) {
                $html .= '<tr>';
                $group = viewHelper::flatEllipsizeText($this->preprocessQuestionText($questionDetails['group']), true, 150);
                $question = viewHelper::flatEllipsizeText($this->preprocessQuestionText($questionDetails['question']), true, 150);
                if ($summaryData['indexMode'] == 'G') {
                    $extraSubmitData = [
                        'cc_target_question' => $questionDetails['qid']
                    ];
                    $html .= '<td>' . $this->getIndexLink($questionDetails['gid'], $group, $extraSubmitData) . '</td>';
                } else {
                    $html .= '<td>' . $group . '</td>';
                }
                if ($summaryData['indexMode'] == 'Q') {
                    $html .= '<td>' . $this->getIndexLink($questionDetails['qid'], $question) . '</td>';
                } else {
                    $html .= '<td>' . $question . '</td>';
                }
                $html .= '</tr>';
            }
            $html .= '</tbody></table></div>';
        } else {
            $description = $this->getQuestionAttribute('summaryCompleteDescription', null, $this->translate('question_attribute.complete_message.default'));
            $html .=
                '<div class="summary-complete">' .
                    '<div class="summary-complete-description">' .$description . '</div>' .
                '</div>';
        }

        return $html;
    }

    private function preprocessQuestionText($text)
    {
        /*$breakOnTokens = ['<span class="popupContainer"'];
        foreach ($breakOnTokens as $token) {
            $pos = stripos($text, $token);
            if ($pos !== false) {
                $text = substr($text, 0, $pos);
                break;
            }
        }*/
        $startToken = '<span class="popupContainer"';
        $endToken = '<span class="closePopupBtn">x</span></span>';
        $startPos = stripos($text, $startToken);
        while ($startPos !== false) {
            $endPos = stripos($text, $endToken, $startPos);
            if ($endPos !== false) {
                $text = substr($text, 0, $startPos) . substr($text, $endPos + strlen($endToken));
            } else {
                $text = substr($text, 0, $startPos);
            }
            $startPos = stripos($text, $startToken);
        }
        return $text;
    }

    private function getIndexLink($itemId, $label, $extraData = null)
    {
        $indexItems = $this->getIndexItems();
        if (empty($indexItems[$itemId])) {
            $out = $label;
        } else {
            $url = $indexItems[$itemId]['url'];
            $submitData = json_decode($indexItems[$itemId]['submit'], true);
            if (!empty($extraData) && is_array($extraData)) {
                $submitData = array_merge($submitData, $extraData);
            }
            $out = '<a href="' . $url . '" data-limesurvey-submit="' . htmlspecialchars(json_encode($submitData)) . '">' . $label . '</a>';
        }
        return $out;
    }

    private function getIndexItems()
    {
        if (empty($this->indexItems)) {
            $fullIndex = LimeSurvey\Helpers\questionIndexHelper::getInstance()->getIndexItems();
            $indexItems = [];
            foreach ($fullIndex as $indexItem) {
                if (isset($indexItem['questions'])) {
                    foreach ($indexItem['questions'] as $question) {
                        $indexItems[$question['qid']] = $question;
                    }
                } else {
                    $indexItems[$indexItem['gid']] = $indexItem;
                }
            }
            $this->indexItems = $indexItems;
        }
        return $this->indexItems;
    }

    private function getProgressHtml($summaryData)
    {
        $percent = (round($summaryData['progress'] * 100)) . "%";
        $description = $this->getQuestionAttribute('summaryProgressDescription', null, $this->translate('question_attribute.progress_description.default'));
        $html =
            '<div class="summary-progress">' .
                '<div class="summary-progress-description">' .$description . '</div>' .
                '<div class="progress">' .
                    '<div class="progress-bar" style="min-width: 2em; width: ' . $percent . ';">' . $percent . '</div>' .
                '</div>' .
            '</div>';
        return $html;
    }

    private function getQuestionAttribute($attribute, $qid = null, $default = null, $language = null)
    {
        if (empty($language)) {
            $language = Yii::app()->language;
        }
        if (empty($qid)) {
            if (empty($this->summaryQid)) {
                return $default;
            }
            $qid = $this->summaryQid;
        }
        if (!isset($this->questionAttributes[$qid])) {
            $this->questionAttributes[$qid] = QuestionAttribute::model()->getQuestionAttributes($qid);
        }
        if (empty($this->questionAttributes[$qid][$attribute])) {
            return $default;
        }
        $value = $this->questionAttributes[$qid][$attribute];
        if (is_array($value) && isset($value[$language])) {
            $value = $value[$language];
        }
        if (!empty($value)) {
            return $value;
        }
        return $default;
    }

    private function getIncompleteScript()
    {
        return "<script>$(document).ready(function(){
            $('#ls-button-submit').prop('disabled', true);
        });</script>";
    }

    private function getExcludedQuestions($qid)
    {
        $excludedSetting = $this->getQuestionAttribute('summaryExcludeQuestions', $qid, '');
        $excludedQuestions = [];
        if (!empty($excludedSetting)) {
            $excludedQuestions = array_map(
                function ($item) {
                    return strtolower(trim($item));
                },
                explode(",", $excludedSetting)
            );
        }
        return $excludedQuestions;
    }

    public function beforeSurveyPage()
    {
        $surveyId = $this->getEvent()->get('surveyId');
        $this->showLink($surveyId);
    }

    private function showLink($surveyId)
    {
        if (!$this->shouldShowLink($surveyId)) {
            return;
        }

        if (!isset($_SESSION['survey_' . $surveyId]['srid'])) {
            return;
        }
        $responseId = $_SESSION['survey_' . $surveyId]['srid'];

        $targetUrl = $this->api->createUrl('plugins/direct', [
            'plugin' => $this->getName(),
            'function' => 'check',
            'surveyId' => $surveyId,
            'srid' => $responseId
        ]);
        $jsVars = [
            'linkLabel' => $this->translate("navbar.link.text"),
            'modalHtml' => $this->getModalHtml(),
            'modalUrl' => $targetUrl,
        ];

        // Used when navigating to a specific question
        $targetQuestion = Yii::app()->getRequest()->getParam('cc_target_question');
        if (!empty($targetQuestion)) {
            $jsVars['targetQuestion'] = $targetQuestion;
        }

        Yii::app()->getClientScript()->registerScript(self::$name, self::$name . '_params = ' . json_encode($jsVars), CClientScript::POS_END);
        Yii::app()->getClientScript()->registerScriptFile($this->getAssetUrl('js') . self::$name . '.js');
        Yii::app()->getClientScript()->registerCssFile($this->getAssetUrl('css') . self::$name . '.css');
    }

    private function shouldShowLink($surveyId)
    {
        return $this->get("showLink", "Survey", $surveyId, false);
    }

    private function getModalHtml()
    {
        ob_start();
        ?>
        <div id="completeness-checker-modal" class="modal fade" role="dialog">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header panel-heading">
                        <button type="button" class="close" data-dismiss="modal">&times;</button>
                        <div class="h4 modal-title">
                            <span class="sr-only"><?= $this->translate("modal.title"); ?></span>
                        </div>
                    </div>
                    <div class="modal-body">
                        <div id="completeness-checker-loader" aria-hidden="true">
                            <div class="completeness-checker-spinner-wrapper">
                                <i class="loader fa fa-circle-o-notch fa-spin fa-3x fa-fw text-muted"></i>
                            </div>
                        </div>
                        <p class='modal-body-text'></p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">&nbsp;<?php eT("Close"); ?></button>
                    </div>
                </div>
            </div>
        </div>
        <?php
        $html = ob_get_contents();
        ob_end_clean();
        return $html;
    }

    private function getSummaryQuestions($surveyId)
    {
        if (empty($this->summaryQuestions[$surveyId])) {
            $questions = [];
            $attributes = QuestionAttribute::model()
                ->resetScope()
                ->with([
                    'qid' => [
                        'select' => 'qid',
                        'together' => true,
                        'joinType' => 'INNER JOIN ',
                    ]
                ])
                ->together()
                ->findAll('qid.sid = :sid AND t.attribute = :attribute AND value = :value', [
                    ':sid' => $surveyId,
                    ':attribute' => 'includeSummary',
                    ':value' => '1'
                ]);
            if (!empty($attributes)) {
                foreach ($attributes as $row) {
                    $questions[] = $row->getRelated('qid')['qid'];
                }
            }
            $this->summaryQuestions[$surveyId] = $questions;
        }
        return $this->summaryQuestions[$surveyId];
    }

    private function getSummaryQuestion($surveyId)
    {
        $summaryQuestions = $this->getSummaryQuestions($surveyId);
        return !empty($summaryQuestions) ? $summaryQuestions[0] : null;
    }
}